<?php

namespace App\Http\Controllers;

use App\Movie;
use App\MovieComment;
use App\MovieSubcomment;
use App\TvSeries;
use App\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class AdminCommentController extends Controller
{
    public function __construct()
    {
        $this->middleware('permission:comment-settings.comments', ['only' => ['index', 'destroy', 'bulk_delete']]);
        $this->middleware('permission:comment-settings.subcomments', ['only' => ['subcommentindex', 'subcommentdestroy', 'sub_bulk_delete']]);
    }
    public function index(Request $request)
    {
        $comment = MovieComment::withCount('subcomments')
            ->where(function ($query) {
                $query->whereNotNull('movie_id')
                    ->orWhereNotNull('tv_series_id');
            })
            ->get();
        $sub_com = MovieSubcomment::all();
        if ($request->ajax()) {
            return \Datatables::of($comment)
                ->addIndexColumn()
                ->addColumn('checkbox', function ($row) {
                    $html = '<div class="inline">
                    <input type="checkbox" form="bulk_delete_form" class="filled-in material-checkbox-input" name="checked[]" value="' . $row->id . '" id="checkbox' . $row->id . '">
                    <label for="checkbox' . $row->id . '" class="material-checkbox"></label>
                </div>';

                    return $html;
                })

                ->addColumn('username', function ($row) {
                    return $row->name;
                })
                ->addColumn('name', function ($row) {
                    if ($row->movie_id) {
                        $movie = Movie::find($row->movie_id);
                        return $movie->title ;
                    } else if ($row->tv_series_id) {
                        $tv = Tvseries::find($row->tv_series_id);
                        return $tv->title;
                    } else {
                        return 'N/A';
                    }
                })
                ->addColumn('comment', function ($row) {
                    $badgeHtml = $row->subcomments_count > 0 
                        ? ' <span class="badge badge-info">' . $row->subcomments_count . '</span>' 
                        : '';
                    return '<button class="btn btn-link comment-btn" data-comment="' . 
                           htmlspecialchars($row->comment, ENT_QUOTES) . '" data-id="' . $row->id . '">' . 
                           (strlen($row->comment) > 50 ? substr($row->comment, 0, 50) . '...' : $row->comment) . 
                           '</button>' . $badgeHtml;
                })
                ->addColumn('status', function ($row) {
                    if ($row->status == 1) {
                        return "<a href=" . route('quick.comment.status', $row->id) . " class='badge badge-pill badge-success'>" . __('Approved') . "</a>";
                    } else {
                        return "<a href=" . route('quick.comment.status', $row->id) . " class='badge badge-pill badge-danger'>" . __('Un Approved') . "</a>";
                    }
                })
                ->addColumn('created_at', function ($row) {
                    return date('F d, Y', strtotime($row->created_at));
                })

                ->addColumn('action', function ($row) {

                    $btn = ' <div class="admin-table-action-block">
                    <button type="button" class="btn btn-round btn-outline-danger" data-target="#deleteModal' . $row->id . '"><i class="fa fa-trash"></i> </button></div>';
                    $btn .= '<div id="deleteModal' . $row->id . '" class="delete-modal modal fade" role="dialog">
                    <div class="modal-dialog modal-sm">
                      <!-- Modal content-->
                      <div class="modal-content">
                        <div class="modal-header">
                          <button type="button" class="close" data-dismiss="modal">&times;</button>
                          <div class="delete-icon"></div>
                        </div>
                        <div class="modal-body text-center">
                          <h4 class="modal-heading">' . __('Are You Sure ?') . '</h4>
                          <p>' . __('Do you really want to delete selected item names here? This
                          process
                          cannot be undone.') . '</p>
                        </div>
                        <div class="modal-footer">
                          <form method="POST" action="' . route("comments.destroy", $row->id) . '">
                            ' . method_field("DELETE") . '
                            ' . csrf_field() . '
                              <button type="reset" class="btn btn-gray translate-y-3" data-dismiss="modal">' . __('No') . '</button>
                              <button type="submit" class="btn btn-danger">' . __('Yes') . '</button>
                          </form>
                        </div>
                      </div>
                    </div>
                  </div>';

                    return $btn;
                })
                ->rawColumns(['checkbox', 'username', 'name', 'comment', 'status', 'created_at', 'action'])
                ->make(true);
        }
        return view('admin.comment.index', compact('comment','sub_com'));
    }

    public function getSubComments(Request $request)
    {
        $commentId = $request->input('comment_id');
        $subComments = MovieSubcomment::with('user')->where('comment_id', $commentId)->get();
        return response()->json(['subComments' => $subComments]);
    }

    public function destroy($id)
    {
        if (env('DEMO_LOCK') == 1) {
            return back()->with('deleted', __('This action is disabled in the demo !'));
        }
        $query = MovieComment::find($id);
        if (isset($query)) {
            $query->delete();

            return back()->with('deleted', __('Comment has been deleted!'));
        } else {
            return back()->with('deleted', __('Comment not found!'));
        }

    }

    public function bulk_delete(Request $request)
    {
        if (env('DEMO_LOCK') == 1) {
            return back()->with('deleted', __('This action is disabled in the demo !'));
        }
        if (env('DEMO_LOCK') == 1) {
            return back()->with('deleted', __('This action is disabled in the demo !'));
        }
        $validator = Validator::make($request->all(), ['checked' => 'required']);

        if ($validator->fails()) {

            return back()->with('deleted', __('Please select one of them to delete'));
        }

        foreach ($request->checked as $checked) {

            $query = MovieComment::findOrFail($checked);
            if (isset($query->subcomment)) {
                foreach ($query->subcomment as $sub) {
                    $sub->delete();
                }
            }
            $query->delete();
        }

        return back()->with('deleted',__('Comment has been deleted'));
    }

    public function subcommentIndex(Request $request)
{
    $comment_id = $request->input('comment_id');
    $subcomments = MovieSubcomment::where('comment_id', $comment_id)->get();

    return \Datatables::of($subcomments)
        ->addIndexColumn()
        ->addColumn('checkbox', function ($row) {
            $html = '<div class="inline">
                <input type="checkbox" form="bulk_delete_form" class="filled-in material-checkbox-input" name="checked[]" value="' . $row->id . '" id="checkbox' . $row->id . '">
                <label for="checkbox' . $row->id . '" class="material-checkbox"></label>
              </div>';

            return $html;
        })
        ->addColumn('username', function ($row) {
            $user = User::find($row->user_id);
            return $user->name;
        })
        ->addColumn('reply', function ($row) {
            return $row->reply;
        })
        ->addColumn('created_at', function ($row) {
            return date('F d, Y', strtotime($row->created_at));
        })
        ->addColumn('status', function ($row) {
            if ($row->status == 1) {
                return "<a href=" . route('quick.subcomment.status', $row->id) . " class='badge badge-pill badge-success'>" . __('Approved') . "</a>";
            } else {
                return "<a href=" . route('quick.subcomment.status', $row->id) . " class='badge badge-pill badge-danger'>" . __('Un Approved') . "</a>";
            }
        })
        ->addColumn('action', function ($row) {
            $btn = '<div class="admin-table-action-block">';
            $btn .= '<a href="javascript:void(0)" data-href="' . route("subcomments.destroy", $row->id) . '" class="btn btn-danger btn-sm delete-subcomment"><i class="fa fa-trash"></i></a>';
            $btn .= '</div>';
            return $btn;
        })
        ->rawColumns(['checkbox', 'username', 'reply', 'created_at', 'status', 'action'])
        ->make(true);
}

    public function quickStatusSubcomment($id)
    {
        $subcomment = MovieSubcomment::findOrFail($id);
        $subcomment->status = !$subcomment->status;
        $subcomment->save();
        return response()->json(['success' => true]);
    }

    public function destroySubcomment($id)
    {
        $subcomment = MovieSubcomment::findOrFail($id);
        $subcomment->delete();
        return response()->json(['success' => true]);
    }

    public function subcommentdestroy($id)
    {
        if (env('DEMO_LOCK') == 1) {
            return back()->with('deleted', __('This action is disabled in the demo !'));
        }
        $query = MovieSubcomment::find($id);
        if (isset($query)) {
            $query->delete();

            return back()->with('deleted', __('SubComment has been deleted!'));
        } else {
            return back()->with('deleted', __('SubComment not found!'));
        }

    }

    public function sub_bulk_delete(Request $request)
    {
        if (env('DEMO_LOCK') == 1) {
            return back()->with('deleted', __('This action is disabled in the demo !'));
        }
        $validator = Validator::make($request->all(), ['checked' => 'required']);

        if ($validator->fails()) {

            return back()
                ->with('deleted', __('Please select one of them to delete'));
        }

        foreach ($request->checked as $checked) {

            $query = MovieSubcomment::findOrFail($checked);

            $query->delete();
        }

        return back()->with('deleted', __('SubComment has been deleted'));
    }
}
